/**
 *
 *
 *
 */
var WalletDemoApp = function()
{

	/**
	 * WalletApp Global var
	 * ...
	 * NEM_EPOCH		 ...
	 * HEXEN_CODE_ARRAY	 ... _hexEncodeArray をリネーム
	 * TransactionType	 ... util/TransactionType.js
	 * BASEENC			 ... util/Address.js の一部 , baseenc をリネーム
	 + CURRENT_NETWORK_ID	 ... 追加
	 */
	var NEM_EPOCH 				 = Date.UTC(2015, 2, 29, 0, 6, 25, 0);
	var HEXEN_CODE_ARRAY		 = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'];
	var TransactionType = {
		Transfer: 0x101, // 257
		ImportanceTransfer: 0x801, // 2049
		MultisigModification: 0x1001, // 4097
		MultisigSignature: 0x1002, // 4098
		MultisigTransaction: 0x1004, // 4100
		ProvisionNamespace: 0x2001, // 8193
		MosaicDefinition: 0x4001, // 16385
		MosaicSupply: 0x4002, // 16386
	};
//	var CURRENT_NETWORK_ID		 = 104;
	var CURRENT_NETWORK_ID		 = 94;
	var BASEENC			 = BASEENC || {};
	BASEENC.b32encode	 = function(s)
	{
		/* encodes a string s to base32 and returns the encoded string */
		var alphabet		 = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
		var parts			 = [];
		var quanta			 = Math.floor((s.length / 5));
		var leftover		 = s.length % 5;
		if (leftover != 0)
		{
			for (var i = 0; i < (5-leftover); i++)
			{
				s += '\x00';
			}
			quanta += 1;
		}
		for (i = 0; i < quanta; i++)
		{
			parts.push(alphabet.charAt(s.charCodeAt(i*5) >> 3));
			parts.push(alphabet.charAt( ((s.charCodeAt(i*5) & 0x07) << 2) | (s.charCodeAt(i*5+1) >> 6)));
			parts.push(alphabet.charAt( ((s.charCodeAt(i*5+1) & 0x3F) >> 1) ));
			parts.push(alphabet.charAt( ((s.charCodeAt(i*5+1) & 0x01) << 4) | (s.charCodeAt(i*5+2) >> 4)));
			parts.push(alphabet.charAt( ((s.charCodeAt(i*5+2) & 0x0F) << 1) | (s.charCodeAt(i*5+3) >> 7)));
			parts.push(alphabet.charAt( ((s.charCodeAt(i*5+3) & 0x7F) >> 2)));
			parts.push(alphabet.charAt( ((s.charCodeAt(i*5+3) & 0x03) << 3) | (s.charCodeAt(i*5+4) >> 5)));
			parts.push(alphabet.charAt( ((s.charCodeAt(i*5+4) & 0x1F) )));
		}
		var replace			 = 0;
		if (leftover == 1)
		{
			replace = 6;
		} else if (leftover == 2) {
			replace = 4;
		} else if (leftover == 3) {
			replace = 3;
		} else if (leftover == 4) {
			replace = 1;
		}
		for (i = 0; i < replace; i++)
		{
			parts.pop();
		}
		for (i = 0; i < replace; i++)
		{
			parts.push("=");
		}
		return parts.join("");
	};
	// this is made specifically for our use, deals only with proper strings
	BASEENC.b32decode = function(s) {
		var alphabet		 = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
		var r				 = new ArrayBuffer(s.length * 5 / 8);
		var b				 = new Uint8Array(r);
		for (var j = 0; j < s.length / 8; j++)
		{
			var v			 = [0,0,0,0, 0,0,0,0];
			for (var i = 0; i < 8; ++i)
			{
				v[i]		 = alphabet.indexOf(s[j*8 + i]);
			}
			var i = 0;
			b[j*5 + 0]		 = (v[i + 0] << 3) | (v[i + 1] >> 2);
			b[j*5 + 1]		 = ((v[i + 1] & 0x3) << 6) | (v[i + 2] << 1) | (v[i + 3] >> 4);
			b[j*5 + 2]		 = ((v[i + 3] & 0xf) << 4) | (v[i + 4] >> 1);
			b[j*5 + 3]		 = ((v[i + 4] & 0x1) << 7) | (v[i + 5] << 2) | (v[i + 6] >> 3);
			b[j*5 + 4]		 = ((v[i + 6] & 0x7) << 5) | (v[i + 7]);
		}
		return b;
	};


	/**
	 * words2ua: WalletApp Global function
	 * ...
	 * @param destUa			 ...
	 * @param cryptowords		 ...
	 * @return
	 */
	function words2ua(destUa, cryptowords)
	{
		for (var i = 0; i < destUa.length; i += 4)
		{
			var v			 = cryptowords.words[i / 4];
			if (v < 0)
			{
				v			 += 0x100000000;
			}
			destUa[i]		 = (v >>> 24);
			destUa[i+1]		 = (v >>> 16) & 0xff;
			destUa[i+2]		 = (v  >>> 8) & 0xff;
			destUa[i+3]		 = v & 0xff;
		}
	}

	/**
	 * ua2words: WalletApp Global function
	 * ...
	 * @param ua			 ...
	 * @param uaLength		 ...
	 * @return
	 */
	function ua2words(ua, uaLength)
	{
		var temp			 = [];
		for (var i = 0; i < uaLength; i += 4)
		{
			var x			 = ua[i]*0x1000000 + (ua[i+1] || 0)*0x10000 + (ua[i+2] || 0)* 0x100 + (ua[i+3] || 0);
			temp.push( (x > 0x7fffffff) ?  x - 0x100000000 : x );
		}
		return CryptoJS.lib.WordArray.create(temp, uaLength);
	}


	/**
	 * hashfunc: WalletApp Global function
	 * ...
	 * @param dest			 ...
	 * @param data			 ...
	 * @param dataLength	 ...
	 * @return
	 */
	function hashfunc(dest, data, dataLength)
	{
		var convertedData		 = ua2words(data, dataLength);
		var hash				 = CryptoJS.SHA3(convertedData, { outputLength: 512 });
		words2ua(dest, hash);
	}


	/**
	 * key_derive: WalletApp Global function
	 * ... hashfunc ?
	 * @param shared		 ...
	 * @param salt			 ...
	 * @param sk			 ...
	 * @param pk			 ...
	 * @return
	 */
	function key_derive(shared, salt, sk, pk)
	{
		nacl.lowlevel.crypto_shared_key_hash(shared, pk, sk, hashfunc);
		for (var i = 0; i < salt.length; i++)
		{
			shared[i]			 ^= salt[i];
		}
		// ua2words
		var hash				 = CryptoJS.SHA3(ua2words(shared, 32),
		{
			outputLength: 256
		});
		return hash;
	}


	/**
	 * hashobj: WalletApp Global function
	 * ...
	 * @param ua			 ...
	 * @param uaLength		 ...
	 * @return
	 */
	function hashobj()
	{
		this.sha3		 = CryptoJS.algo.SHA3.create({ outputLength: 512 });
		this.reset		 = function()
		{
			this.sha3	 = CryptoJS.algo.SHA3.create({ outputLength: 512 });
		}
		this.update		 = function(data)
		{
			if (data instanceof BinaryKey)
			{
				var converted		 = ua2words(data.data, data.data.length);
				var result			 = CryptoJS.enc.Hex.stringify(converted);
				this.sha3.update(converted);
			}
			else if (data instanceof Uint8Array)
			{
				var converted		 = ua2words(data, data.length);
				this.sha3.update(converted);
			}
			else if (typeof data === "string")
			{
				var converted		 = CryptoJS.enc.Hex.parse(data);
				this.sha3.update(converted);
			}
			else
			{
				throw new Error("unhandled argument");
			}
		}
		this.finalize		 = function(result)
		{
			var hash			 = this.sha3.finalize();
			words2ua(result, hash);
		};
	}


	/**
	 * BinaryKey
	 * ... this.data
	 * @param keyData		 ...
	 * @return
	 */
	function BinaryKey(keyData)
	{
		var convert			 = new Convert();
		this.data			 = keyData;
		this.toString		 = function() {
			return convert.ua2hex(this.data);
		}
	}


	/**
	 * keyPair
	 * ...
	 * @param privkey		 ...
	 * @return
	 */
	function KEYPAIR(privkey)
	{
		var convert			 = new Convert();
		this.publicKey		 = new BinaryKey(new Uint8Array(nacl.lowlevel.crypto_sign_PUBLICKEYBYTES));
		this.secretKey		 = convert.hex2ua_reversed(privkey);
		nacl.lowlevel.crypto_sign_keypair_hash(this.publicKey.data, this.secretKey, hashfunc);
		this.sign			 = function(data)
		{
			var sig			 = new Uint8Array(64);
			var hasher		 = new hashobj();
			var r			 = nacl.lowlevel.crypto_sign_hash(sig, this, data, hasher);
			if (!r)
			{
				alert("couldn't sign the tx, generated invalid signature");
				throw new Error("couldn't sign the tx, generated invalid signature");
			}
			return new BinaryKey(sig);
		}
	};


	/**
	 * CURRENT_NETWORK_VERSION
	 * ... nem, mijin と固定でIDが振られていたような気がする
	 * ... CURRENT_NETWORK_ID を固定で設定する
	 * @param val		 ...
	 * @return
	 */
	function CURRENT_NETWORK_VERSION(val)
	{
		if (CURRENT_NETWORK_ID === 104)
		{
			return 0x68000000 | val;
		} else if (CURRENT_NETWORK_ID === -104) {
			return 0x98000000 | val;
		}
		return 0x60000000 | val;
	};


	/**
	 * CREATE_DATA
	 * ...
	 * @param txtype			 ...
	 * @param senderPublicKey	 ...
	 * @param timeStamp			 ...
	 * @param due				 ...
	 * @param version			 ...
	 * @return
	 */
	function CREATE_DATA(txtype, senderPublicKey, timeStamp, due, version)
	{
		// マルチシグトランザクションの場合、時間をのばす
		/*
		var addtime = 600;
		if(txtype == 4100)
		{
			addtime		 = addtime * 1000;
		}
		*/
		return {
			'type':			 txtype,
			'version':		 version || CURRENT_NETWORK_VERSION(1),
			'signer':		 senderPublicKey,
			'timeStamp':	 timeStamp,
//			'deadline':		 timeStamp + due * 60
			'deadline':      timeStamp + 60000
//			'deadline':		 timeStamp + due * addtime
		};
	}


	/**
	 * CALC_MIN_FEE
	 * ...
	 * @param numNem		 ...
	 * @return
	 */
	function CALC_MIN_FEE(numNem)
	{
		return Math.ceil(Math.max(10 - numNem, 2, Math.floor(Math.atan(numNem / 150000.0) * 3 * 33)));
	}


	/**
	 * fixPrivateKey
	 * ...
	 * @param privatekey		 ...
	 * @return
	 */
	function fixPrivateKey(privatekey)
	{
		return ("0000000000000000000000000000000000000000000000000000000000000000" + privatekey.replace(/^00/, '')).slice(-64);
	}


	/**
	 * mosaicIdToName
	 * ...
	 * @param mosaicId		 ...
	 * ... mosaicId.namespaceId, mosaicId.name 必須
	 * @return
	 */
	function mosaicIdToName(mosaicId)
	{
		return mosaicId.namespaceId + ":" + mosaicId.name;
	}


	/**
	 * calcXemEquivalent
	 * ...
	 * @param multiplier		 ...
	 * @param q					 ...
	 * @param sup				 ...
	 * @param divisibility		 ...
	 * @return
	 */
	function calcXemEquivalent(multiplier, q, sup, divisibility)
	{
		if (sup === 0)
		{
			return 0;
		}

		// TODO: can this go out of JS (2^54) bounds? (possible BUG)
		return 8999999999 * q * multiplier / sup / Math.pow(10, divisibility + 6);
	}


	/**
	 * Convert
	 * ... utils/convert.js から作成
	 * @return
	 */
	var Convert = function()
	{
		var obj					 = {};

		/**
		 * Convert.hex2ua_reversed
		 * ...
		 * @param hexx ...
		 */
		obj.hex2ua_reversed = function(hexx)
		{
			var hex			 = hexx.toString();//force conversion
			var ua			 = new Uint8Array(hex.length / 2);
			for (var i = 0; i < hex.length; i += 2)
			{
				ua[ua.length - 1 - (i / 2)]		 = parseInt(hex.substr(i, 2), 16);
			}
			return ua;
		};


		/**
		 * Convert.hex2ua
		 * ...
		 * @param hexx ...
		 */
		obj.hex2ua = function(hexx)
		{
			var hex			 = hexx.toString();//force conversion
			var ua			 = new Uint8Array(hex.length / 2);
			for (var i = 0; i < hex.length; i += 2)
			{
				ua[i / 2]		 = parseInt(hex.substr(i, 2), 16);
			}
			return ua;
		};


		/**
		 * Convert.ua2hex
		 * ...
		 * @param hexx ...
		 */
		obj.ua2hex = function(ua)
		{
			var s			 = '';
			for (var i = 0; i < ua.length; i++)
			{
				var code	 = ua[i];
				s			 += HEXEN_CODE_ARRAY[code >>> 4];
				s			 += HEXEN_CODE_ARRAY[code & 0x0F];
			}
			return s;
		};


		/**
		 * Convert.hex2a
		 * ...
		 * @param hexx ...
		 */
		obj.hex2a = function(hexx)
		{
			var hex			 = hexx.toString();
			var str			 = '';
			for (var i = 0; i < hex.length; i += 2)
			{
				str			 += String.fromCharCode(parseInt(hex.substr(i, 2), 16));
			}
			return str;
		};


		/**
		 * Convert.utf8ToHex
		 * ...
		 * @param str ...
		 */
		obj.utf8ToHex = function(str)
		{
			var hex;
			try {
				hex = unescape(encodeURIComponent(str)).split('').map(function(v){
					return v.charCodeAt(0).toString(16)
				}).join('');
			} catch(e){
				hex = str;
				console.log('invalid text input: ' + str);
			}
			return hex;
		};

		return obj;
	}

	/**
	 * TransactionsFactory
	 * ... services/Transactions.js から作成
	 * @param $http		 ...
	 * @param host		 ... ホスト（111.111.111.111 とか）
	 * @return
	 */
	var TransactionsFactory = function(host)
	{
		var obj					 = {};
		// これもいらんかも
		// 固定で設定すればよさげ
		var CURRENT_NETWORK_ID	 = '';
		var CURRENT_HOSTNAME	 = host;


		/**
		 * TransactionsFactory.getTimeStamp
		 * ...
		 * @return
		 */
		obj.getTimeStamp = function()
		{
			return Math.floor((Date.now() / 1000) - (NEM_EPOCH / 1000));
		};


		/**
		 * TransactionsFactory.constructTransfer
		 * ...
		 * @param senderPublicKey			 ...
		 * @param recipientCompressedKey	 ...
		 * @param amount					 ...
		 * @param message					 ...
		 * @param due						 ...
		 * @param mosaics					 ...
		 * @param mosaicsFee				 ...
		 * @return
		 */
		obj.constructTransfer = function(senderPublicKey, recipientCompressedKey, amount, message, due, mosaics, mosaicsFee)
		{
			var timeStamp		 = this.getTimeStamp();
			var version			 = mosaics ? CURRENT_NETWORK_VERSION(2) : CURRENT_NETWORK_VERSION(1);
			var data			 = CREATE_DATA(0x101, senderPublicKey, timeStamp, due, version);
			var msgFee			 = message.payload.length ? Math.max(1, Math.floor(message.payload.length / 2 / 16)) * 2 : 0;
			var fee				 = mosaics ? mosaicsFee : CALC_MIN_FEE(amount / 1000000);
			var totalFee		 = fee * 1000000;
			var custom			 = {
				'recipient': recipientCompressedKey.toUpperCase().replace(/-/g, ''),
				'amount': amount,
				'fee': totalFee,
				'message': message,
				'mosaics': mosaics
			};
			var entity			 = $.extend(data, custom);
			return entity;
		};


		/**
		 * TransactionsFactory.constructConvertMultisigAccount
		 * ...
		 * @param senderPublicKey		 ...
		 * @param modifications			 ...
		 * @param minCosignatories		 ...
		 * @param due					 ...
		 * @return
		 */
		obj.constructMultisigAggregateModification = function(senderPublicKey, modifications, minCosignatories, due)
		{
			var timeStamp		 = this.getTimeStamp();
			var version			 = CURRENT_NETWORK_VERSION(1);
			var data			 = CREATE_DATA(0x1001, senderPublicKey, timeStamp, due, version);
			var fee				 = due;
			var totalFee		 = (fee) * 1000000;
			var _modifications	 = [];
			for (var i = 0; i < modifications.length; i++)
			{
//				var modification	 = {};
				var modification	 = new Object();
				modification.modificationType		 = 1;
				modification.cosignatoryAccount		 = modifications[i];
				_modifications.push(modification);
			}
			var custom = {
				"modifications": _modifications,
				"minCosignatories":		 {
					"relativeChange":		 parseInt(minCosignatories)
				},
				"fee":					 totalFee
			};
			var entity			 = $.extend(data, custom);
			return entity;
		};

		/**
		 * TransactionsFactory.constructNamespace
		 * ...
		 * @param senderPublicKey		 ...
		 * @param rentalFeeSink			 ...
		 * @param rentalFee				 ...
		 * @param namespaceParent		 ...
		 * @param namespaceName			 ...
		 * @param due					 ...
		 * @return
		 */
		obj.constructNamespace = function(senderPublicKey, rentalFeeSink, rentalFee, namespaceParent, namespaceName, due)
		{
			var timeStamp		 = this.getTimeStamp();
			var version			 = CURRENT_NETWORK_VERSION(1);
			var data			 = CREATE_DATA(0x2001, senderPublicKey, timeStamp, due, version);
			var fee				 = 2 * 3 * 18;
			var totalFee		 = (fee) * 1000000;
			var parent			 = (namespaceParent) ? namespaceParent : null;
			var custom			 = {
				'rentalFeeSink': rentalFeeSink.toUpperCase().replace(/-/g, ''),
				'rentalFee': rentalFee,
				'parent': parent,
				'newPart': namespaceName,
				'fee': totalFee
			};
			var entity			 = $.extend(data, custom);
			return entity;
		};


		/**
		 * TransactionsFactory.constructMosaicDefinition
		 * ...
		 * @param senderPublicKey		 ...
		 * @param rentalFeeSink			 ...
		 * @param rentalFee				 ...
		 * @param namespaceParent		 ...
		 * @param mosaicName			 ...
		 * @param mosaicDescription		 ...
		 * @param mosaicProperties		 ...
		 * @param levy					 ...
		 * @param due					 ...
		 * @return
		 */
		obj.constructMosaicDefinition = function(senderPublicKey, rentalFeeSink, rentalFee, namespaceParent, mosaicName, mosaicDescription, mosaicProperties, levy, due)
		{
			var timeStamp		 = this.getTimeStamp();
			var version			 = CURRENT_NETWORK_VERSION(1);
			var data			 = CREATE_DATA(0x4001, senderPublicKey, timeStamp, due, version);
			var fee				 = 2 * 3 * 18;
			var totalFee		 = (fee) * 1000000;
			var levyData		 = levy ? {
										'type':			 levy.feeType,
										'recipient':	 levy.address.toUpperCase().replace(/-/g, ''),
										'mosaicId':		 levy.mosaic,
										'fee':			 levy.fee,
									} : null;
			var custom			 = {
										'creationFeeSink':		 rentalFeeSink.replace(/-/g, ''),
										'creationFee':			 rentalFee,
										'mosaicDefinition':		 {
											'creator':		 senderPublicKey,
											'id': {
												'namespaceId': 	 namespaceParent,
												'name': 		 mosaicName,
											},
											'description':		 mosaicDescription,
											'properties':		 $.map(mosaicProperties, function(v,k){
												return {'name':k, 'value':v.toString()};
											}),
											'levy': levyData
										},
										'fee': totalFee
									};
			var entity			 = $.extend(data, custom);
			return entity;
		};


		/**
		 * TransactionsFactory.constructMosaicSupply
		 * ...
		 * @param senderPublicKey		 ...
		 * @param mosaicId				 ...
		 * @param supplyType			 ...
		 * @param delta					 ...
		 * @param due					 ...
		 * @return
		 */
		obj.constructMosaicSupply = function(senderPublicKey, mosaicId, supplyType, delta, due)
		{
			var timeStamp		 = this.getTimeStamp();
			var version			 = CURRENT_NETWORK_VERSION(1);
			var data			 = CREATE_DATA(0x4002, senderPublicKey, timeStamp, due, version);
			var fee				 = 2 * 3 * 18;
			var totalFee		 = (fee) * 1000000;
			var custom = {
				'mosaicId':		 mosaicId,
				'supplyType':	 supplyType,
				'delta':		 delta,
				'fee':			 totalFee
			};
			var entity			 = $.extend(data, custom);
			return entity;
		};


		/**
		 * TransactionsFactory.constructSignature
		 * ...
		 * @param senderPublicKey		 ...
		 * @param otherAccount			 ...
		 * @param otherHash				 ...
		 * @param due					 ...
		 * @return
		 */
		obj.constructSignature = function(senderPublicKey, otherAccount, otherHash, due)
		{
			var timeStamp		 = this.getTimeStamp();
			var version			 = CURRENT_NETWORK_VERSION(1);
			var data			 = CREATE_DATA(0x1002, senderPublicKey, timeStamp, due, version);
			var totalFee		 = (2 * 3) * 1000000;
			var custom = {
				'otherHash':	 { 'data': otherHash },
				'otherAccount':	 otherAccount,
				'fee':			 totalFee,
			};
			var entity			 = $.extend(data, custom);
			return entity;
		};


		/**
		 * TransactionsFactory.multisigWrapper
		 * ...
		 * @param senderPublicKey		 ...
		 * @param innerEntity			 ...
		 * @param due					 ...
		 * @return
		 */
		obj.multisigWrapper = function(senderPublicKey, innerEntity, due)
		{
			var timeStamp		 = this.getTimeStamp();
			var version			 = CURRENT_NETWORK_VERSION(1);
			var data			 = CREATE_DATA(0x1004, senderPublicKey, timeStamp, due, version);
			var custom = {
//				'signatures':	 [],
				'fee':			 18000000,
				'otherTrans':	 innerEntity
			};
			var entity			 = $.extend(data, custom);
			return entity;
		};


		/**
		 * TransactionsFactory.serializeSafeString
		 * ...
		 * @param str				 ...
		 * @return
		 */
		/**
		 * NOTE, related to serialization: Unfortunately we need to create few objects
		 * and do a bit of copying, as Uint32Array does not allow random offsets
		 */
		/* safe string - each char is 8 bit */
		obj.serializeSafeString = function(str)
		{
			var r			 = new ArrayBuffer(132);
			var d			 = new Uint32Array(r);
			var b			 = new Uint8Array(r);
			var e			 = 4;
			if (str === null)
			{
				d[0]		 = 0xffffffff;

			} else {
				d[0]		 = str.length;
				for (var j = 0; j < str.length; ++j)
				{
					b[e++]	 = str.charCodeAt(j);
				}
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeUaString
		 * ...
		 * @param str				 ...
		 * @return
		 */
		obj.serializeUaString = function(str)
		{
			var r		 = new ArrayBuffer(516);
			var d		 = new Uint32Array(r);
			var b		 = new Uint8Array(r);
			var e		 = 4;
			if (str === null)
			{
				d[0] = 0xffffffff;

			} else {
				d[0]	 = str.length;
				for (var j = 0; j < str.length; ++j)
				{
					b[e++]	 = str[j];
				}
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeLong
		 * ...
		 * @param value				 ...
		 * @return
		 */
		obj.serializeLong = function(value)
		{
			var r 		 = new ArrayBuffer(8);
			var d		 = new Uint32Array(r);
			d[0]		 = value;
			d[1]		 = Math.floor((value / 0x100000000));
			return new Uint8Array(r, 0, 8);
		};


		/**
		 * TransactionsFactory.serializeMosaicId
		 * ...
		 * @param mosaicId			 ...
		 * @return
		 */
		obj.serializeMosaicId = function(mosaicId)
		{
			var r						 = new ArrayBuffer(264);
			var serializedNamespaceId	 = this.serializeSafeString(mosaicId.namespaceId);
			var serializedName			 = this.serializeSafeString(mosaicId.name);
			var b						 = new Uint8Array(r);
			var d						 = new Uint32Array(r);
			d[0]						 = serializedNamespaceId.length + serializedName.length;
			var e						 = 4;
			for (var j=0; j<serializedNamespaceId.length; ++j)
			{
				b[e++]			 = serializedNamespaceId[j];
			}
			for (var j = 0; j < serializedName.length; ++j)
			{
				b[e++]			 = serializedName[j];
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeMosaicAndQuantity
		 * ...
		 * @param mosaic			 ...
		 * @return
		 */
		obj.serializeMosaicAndQuantity = function(mosaic)
		{
			var r					 = new ArrayBuffer(4 + 264 + 8);
			var serializedMosaicId	 = this.serializeMosaicId(mosaic.mosaicId);
			var serializedQuantity	 = this.serializeLong(mosaic.quantity);
			var b					 = new Uint8Array(r);
			var d					 = new Uint32Array(r);
			d[0]					 = serializedMosaicId.length + serializedQuantity.length;
			var e					 = 4;
			for (var j=0; j<serializedMosaicId.length; ++j)
			{
				b[e++]				 = serializedMosaicId[j];
			}
			for (var j=0; j<serializedQuantity.length; ++j)
			{
				b[e++]				 = serializedQuantity[j];
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeMosaics
		 * ...
		 * @param entity			 ...
		 * @return
		 */
		obj.serializeMosaics = function(entity)
		{
			var r			 = new ArrayBuffer(276*10 + 4);
			var d			 = new Uint32Array(r);
			var b			 = new Uint8Array(r);

			var i			 = 0;
			var e			 = 0;
			d[i++]			 = entity.length;
			e += 4;
			var temporary	 = [];

			for (var j=0; j<entity.length; ++j)
			{
				temporary.push({'entity':entity[j], 'value':mosaicIdToName(entity[j].mosaicId) + " : " + entity[j].quantity})
			}
			temporary.sort(function(a, b) {return a.value < b.value ? -1 : a.value > b.value;});

			for (var j=0; j<temporary.length; ++j)
			{
				var entity				 = temporary[j].entity;
				var serializedMosaic	 = this.serializeMosaicAndQuantity(entity);
				for (var k=0; k<serializedMosaic.length; ++k)
				{
					b[e++]				 = serializedMosaic[k];
				}
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeProperty
		 * ...
		 * @param entity			 ...
		 * @return
		 */
		obj.serializeProperty = function(entity)
		{
			var r					 = new ArrayBuffer(1024);
			var d					 = new Uint32Array(r);
			var b					 = new Uint8Array(r);
			var serializedName		 = this.serializeSafeString(entity['name']);
			var serializedValue		 = this.serializeSafeString(entity['value']);
			d[0]					 = serializedName.length + serializedValue.length;
			var e					 = 4;
			for (var j = 0; j<serializedName.length; ++j)
			{
				b[e++]		 = serializedName[j];
			}
			for (var j = 0; j<serializedValue.length; ++j)
			{
				b[e++]		 = serializedValue[j];
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeProperties
		 * ...
		 * @param entity			 ...
		 * @return
		 */
		obj.serializeProperties = function(entity)
		{
			var r			 = new ArrayBuffer(1024);
			var d			 = new Uint32Array(r);
			var b			 = new Uint8Array(r);
			var i			 = 0;
			var e			 = 0;
			d[i++]			 = entity.length;
			e				 += 4;
			var temporary	 = entity;
			var temporary	 = [];

			for (var j=0; j<entity.length; ++j)
			{
				temporary.push(entity[j]);
			}
			var helper		 = {
									'divisibility':1,
									'initialSupply':2,
									'supplyMutable':3,
									'transferable':4
								};
			temporary.sort(function(a, b)
			{
				return helper[a.name] < helper[b.name] ? -1 : helper[a.name] > helper[b.name];
			});
			for (var j=0; j<temporary.length; ++j)
			{
				var entity					 = temporary[j];
				var serializedProperty		 = this.serializeProperty(entity);
				for (var k=0; k<serializedProperty.length; ++k)
				{
					b[e++]					 = serializedProperty[k];
				}
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeLevy
		 * ...
		 * @param entity			 ...
		 * @return
		 */
		obj.serializeLevy = function(entity)
		{
			var r			 = new ArrayBuffer(1024);
			var d			 = new Uint32Array(r);
			if (entity === null)
			{
				d[0]		 = 0;
				return new Uint8Array(r, 0, 4);
			}
			var b			 = new Uint8Array(r);
			d[1]			 = entity['type'];
			var e			 = 8;
			var temp		 = this.serializeSafeString(entity['recipient']);
			for (var j = 0; j<temp.length; ++j)
			{
				b[e++]		 = temp[j];
			}
			var serializedMosaicId		 = this.serializeMosaicId(entity['mosaicId']);
			for (var j=0; j<serializedMosaicId.length; ++j)
			{
				b[e++]			 = serializedMosaicId[j];
			}
			var serializedFee	 = this.serializeLong(entity['fee']);
			for (var j=0; j<serializedFee.length; ++j)
			{
				b[e++]			 = serializedFee[j];
			}
			d[0]				 = 4 + temp.length + serializedMosaicId.length + 8;
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeMosaicDefinition
		 * ...
		 * @param entity			 ...
		 * @return
		 */
		obj.serializeMosaicDefinition = function(entity)
		{
			var convert		 = new Convert();
			var r			 = new ArrayBuffer(40 + 264 + 516 + 1024 + 1024);
			var d			 = new Uint32Array(r);
			var b			 = new Uint8Array(r);
			var temp		 = convert.hex2ua(entity['creator']);
			d[0]			 = temp.length;
			var e			 = 4;
			for (var j = 0; j<temp.length; ++j)
			{
				b[e++]		 = temp[j];
			}
			var serializedMosaicId	 = this.serializeMosaicId(entity.id);
			for (var j = 0; j < serializedMosaicId.length; ++j)
			{
				b[e++]		 = serializedMosaicId[j];
			}
			var utf8ToUa	 = convert.hex2ua(convert.utf8ToHex(entity['description']));
			var temp		 = this.serializeUaString(utf8ToUa);
			for (var j = 0; j < temp.length; ++j)
			{
				b[e++]		 = temp[j];
			}
			var temp		 = this.serializeProperties(entity['properties']);
			for (var j = 0; j < temp.length; ++j)
			{
				b[e++]		 = temp[j];
			}
			var levy		 = this.serializeLevy(entity['levy']);
			for (var j = 0; j < levy.length; ++j)
			{
				b[e++]		 = levy[j];
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.serializeTransaction
		 * ...
		 * @param entity			 ...
		 * @return
		 */
		obj.serializeTransaction = function(entity)
		{
			var convert		 = new Convert();
			var r			 = new ArrayBuffer(512 + 2764);
			var d			 = new Uint32Array(r);
			var b			 = new Uint8Array(r);
			d[0]			 = entity['type'];
			d[1]			 = entity['version'];
			d[2]			 = entity['timeStamp'];
			var temp		 = convert.hex2ua(entity['signer']);
			d[3]			 = temp.length;
			var e			 = 16;
			for (var j = 0; j < temp.length; ++j)
			{
				b[e++]		 = temp[j];
			}
			// Transaction
			var i			 = e / 4;
			d[i++]			 = entity['fee'];
			d[i++]			 = Math.floor((entity['fee'] / 0x100000000));
			d[i++]			 = entity['deadline'];
			e				 += 12;


			// TransferTransaction
			if (d[0] === TransactionType.Transfer)
			{
				d[i++]			 = entity['recipient'].length;
				e				 += 4;

				// TODO: check that entity['recipient'].length is always 40 bytes
				for (var j = 0; j < entity['recipient'].length; ++j)
				{
					b[e++]		 = entity['recipient'].charCodeAt(j);
				}
				i				 = e / 4;
				d[i++]			 = entity['amount'];
				d[i++]			 = Math.floor((entity['amount'] / 0x100000000));
				e				 += 8;
				if (entity['message']['type'] === 1 || entity['message']['type'] === 2)
				{
					var temp		 = convert.hex2ua(entity['message']['payload']);
					if (temp.length === 0)
					{
						d[i++]		 = 0;
						e			 += 4;
					} else {
						// length of a message object
						d[i++]		 = 8 + temp.length;
						// object itself
						d[i++]		 = entity['message']['type'];
						d[i++]		 = temp.length;
						e			 += 12;
						for (var j = 0; j < temp.length; ++j)
						{
							b[e++]	 = temp[j];
						}
					}
				}
				var entityVersion	 = d[1] & 0xffffff;
				if (entityVersion >= 2)
				{
					var temp		 = this.serializeMosaics(entity['mosaics']);
					for (var j = 0; j < temp.length; ++j)
					{
						b[e++]		 = temp[j];
					}
				}
			// Provision Namespace transaction
			} else if (d[0] === TransactionType.ProvisionNamespace) {
				d[i++]			 = entity['rentalFeeSink'].length;
				e				 += 4;
				// TODO: check that entity['rentalFeeSink'].length is always 40 bytes
				for (var j = 0; j < entity['rentalFeeSink'].length; ++j)
				{
					b[e++]		 = entity['rentalFeeSink'].charCodeAt(j);
				}
				i				 = e / 4;
				d[i++]			 = entity['rentalFee'];
				d[i++]			 = Math.floor((entity['rentalFee'] / 0x100000000));
				e				 += 8;
				var temp		 = this.serializeSafeString(entity['newPart']);
				for (var j = 0; j < temp.length; ++j)
				{
					b[e++]		 = temp[j];
				}
				var temp		 = this.serializeSafeString(entity['parent']);
				for (var j = 0; j<temp.length; ++j)
				{
					b[e++]		 = temp[j];
				}
			// Mosaic Definition Creation transaction
			} else if (d[0] === TransactionType.MosaicDefinition) {
				var temp		 = this.serializeMosaicDefinition(entity['mosaicDefinition']);
				d[i++]			 = temp.length;
				e				 += 4;
				for (var j = 0; j < temp.length; ++j)
				{
					b[e++]		 = temp[j];
				}
				temp			 = this.serializeSafeString(entity['creationFeeSink']);
				for (var j = 0; j < temp.length; ++j)
				{
					b[e++]		 = temp[j];
				}
				temp			 = this.serializeLong(entity['creationFee']);
				for (var j = 0; j < temp.length; ++j)
				{
					b[e++]		 = temp[j];
				}
			// Mosaic Supply Change transaction
			} else if (d[0] === TransactionType.MosaicSupply) {
				var serializedMosaicId		 = this.serializeMosaicId(entity['mosaicId']);
				for (var j=0; j<serializedMosaicId.length; ++j)
				{
					b[e++]			 = serializedMosaicId[j];
				}

				var temp					 = new ArrayBuffer(4);
				d							 = new Uint32Array(temp);
				d[0]						 = entity['supplyType'];
				var serializeSupplyType		 = new Uint8Array(temp);
				for (var j=0; j < serializeSupplyType.length; ++j)
				{
					b[e++]				 = serializeSupplyType[j];
				}

				var serializedDelta		 = this.serializeLong(entity['delta']);
				for (var j=0; j<serializedDelta.length; ++j)
				{
					b[e++]				 = serializedDelta[j];
				}
			// MultisigModification
			} else if (d[0] === TransactionType.MultisigModification) {
				var temp		 = entity['modifications'];
				d[i++]			 = temp.length;
				e				 += 4;
				for (var j = 0; j < temp.length; ++j)
				{
					// Length of modification structure
					d[i++]			 = 0x28;
					e				 += 4;
					// Modification type
					if (temp[j]['modificationType'] == 1)
					{
						d[i++]		 = 0x01;
					} else {
						d[i++]		 = 0x02;
					}
					e				 += 4;
					// Length of public key
					d[i++]			 = 0x20;
					e				 += 4;
					var key2bytes	 = convert.hex2ua(temp[j]['cosignatoryAccount']);
					// Key to Bytes
					for (var k = 0; k < key2bytes.length; ++k)
					{
						b[e++]		 = key2bytes[k];
					}
					i				 = e / 4;
				}
				var entityVersion		 = d[1] & 0xffffff;
				if (entityVersion >= 2)
				{
					d[i++]		 = 0x04;
					e			 += 4;
					// Relative change
					d[i++]		 = entity['minCosignatories']['relativeChange'].toString(16);
					e += 4;
				} else {
					// Version 1 has no modifications
				}
			// Multisig wrapped transaction
			} else if (d[0] === TransactionType.MultisigSignature) {
				var temp		 = convert.hex2ua(entity['otherHash']['data']);
				// length of a hash object....
				d[i++]			 = 4 + temp.length;
				// object itself
				d[i++]			 = temp.length;
				e				 += 8;
				for (var j = 0; j < temp.length; ++j)
				{
					b[e++]		 = temp[j];
				}
				i				 = e / 4;
				temp			 = entity['otherAccount'];
				d[i++]			 = temp.length;
				e				 += 4;
				for (var j = 0; j < temp.length; ++j)
				{
					b[e++]		 = temp.charCodeAt(j);
				}
			// Multisig wrapped transaction
			} else if (d[0] === TransactionType.MultisigTransaction){
				var temp		 = this.serializeTransaction(entity['otherTrans']);
				d[i++]			 = temp.length;
				e				 += 4;
				for (var j = 0; j < temp.length; ++j)
				{
					b[e++]		 = temp[j];
				}
			}
			return new Uint8Array(r, 0, e);
		};


		/**
		 * TransactionsFactory.prepareMessage
		 * ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @return
		 */
		obj.prepareMessage = function(privatekey, tx)
		{
			var convert				 = new Convert();
			var cryptoHelpers		 = new CryptoHelpers();
			var message				 = (tx.message) ? tx.message.toString() : '';
			if (tx.encryptMessage)
			{
				if (!tx.recipientPubKey || !tx.message || !privatekey)
				{
					return {
						'type':0,
						'payload':''
					};
				}
				return {
					'type':2,
					'payload':cryptoHelpers.encode(privatekey, tx.recipientPubKey, message)
				};
			}
			return {
				'type': 1,
				'payload':convert.utf8ToHex(message)
			}
		};


		/**
		 * TransactionsFactory.prepareTransfer
		 * ...
		 * @param publickey			 ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @return
		 */
		obj.prepareTransfer = function(publickey, privatekey, tx)
		{
			var keyPair			 = new KeyPair();
			var kp				 = keyPair.create(privatekey);
			var actualSender	 = tx.isMultisig ? tx.multisigAccount.publicKey : kp.publicKey.toString();
			var recipientCompressedKey		 = (tx.recipient) ? tx.recipient.toString() : '';
			var amount						 = parseInt(tx.amount * 1000000, 10);
			var message						 = this.prepareMessage(privatekey, tx);
			var due							 = tx.due;
			var mosaics						 = null;
			var mosaicsFee					 = null;
			var entity						 = this.constructTransfer(actualSender, recipientCompressedKey, amount, message, due, mosaics, mosaicsFee);
			if (tx.isMultisig)
			{
				entity		 = this.multisigWrapper(kp.publicKey.toString(), entity, due);
			}
			return entity;
		};


		/**
		 * TransactionsFactory.calculateMosaicsFee
		 * ...
		 * @param multiplier			 ...
		 * @param mosaics				 ...
		 * @param attachedMosaics		 ...
		 * @return
		 */
		obj.calculateMosaicsFee = function(multiplier, mosaics, attachedMosaics)
		{
			var totalFee			 = 0;
			for (var m of attachedMosaics)
			{
				// TODO: copied from filters, refactor
				var mosaicName			 = mosaicIdToName(m.mosaicId);
				if (!(mosaicName in mosaics))
				{
					return ['unknown mosaic divisibility', data];
				}
				var mosaicDefinitionMetaDataPair	 = mosaics[mosaicName];
				var divisibilityProperties			 = $.grep(mosaicDefinitionMetaDataPair.mosaicDefinition.properties, function(w){
						return w.name === "divisibility";
					});
				var divisibility	 = divisibilityProperties.length === 1 ? ~~(divisibilityProperties[0].value) : 0;

				//var supply = mosaicDefinitionMetaDataPair.meta.supply;
				var supply			 = mosaicDefinitionMetaDataPair.supply;
				var quantity		 = m.quantity;
				var numNem			 = calcXemEquivalent(multiplier, quantity, supply, divisibility);
				var fee				 = CALC_MIN_FEE(numNem);

				 //console.log("CALCULATING FEE for ", m, mosaicDefinitionMetaDataPair, "divisibility", divisibility, "nem equivalent", numNem, "calculated fee", fee);
				totalFee			 += fee;
			}
			return (totalFee * 5) / 4;
		};


		/**
		 * TransactionsFactory.prepareTransferV2
		 * ...
		 * @param privatekey			 ...
		 * @param mosaicsMetaData		 ...
		 * @param tx					 ...
		 * @return
		 */
		obj.prepareTransferV2 = function(privatekey, tx)
		{
			var keyPair				 = new KeyPair();
			var kp					 = keyPair.create(privatekey);
			var actualSender		 = tx.isMultisig ? tx.multisigAccount.publicKey : kp.publicKey.toString();
//			var actualSender		 = tx.isMultisig ? tx.publicKey : kp.publicKey.toString();
			var recipientCompressedKey		 = tx.recipient.toString();
			// multiplier
			var amount				 = parseInt(tx.multiplier * 1000000, 10);
//			var amount				 = parseInt(tx.multiplier * 1, 10);
			var message				 = this.prepareMessage(privatekey, tx);
			var due					 = tx.due;
			var mosaics				 = tx.mosaics;
//			var mosaicsFee			 = this.calculateMosaicsFee(amount, mosaicsMetaData, mosaics);
			var mosaicsFee			 = 1000;
			var entity				 = this.constructTransfer(actualSender, recipientCompressedKey, amount, message, due, mosaics, mosaicsFee);
			if (tx.isMultisig)
			{
				entity = this.multisigWrapper(kp.publicKey.toString(), entity, due);
			}
			return entity;
		};


		/**
		 * TransactionsFactory.prepareNamespace
		 * ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @return
		 */
		obj.prepareNamespace = function(privatekey, tx)
		{
			var keyPair				 = new KeyPair();
			var kp					 = keyPair.create(privatekey);
			var actualSender		 = tx.isMultisig ? tx.multisigAccount.publicKey : kp.publicKey.toString();
			var rentalFeeSink		 = tx.rentalFeeSink.toString();
			var rentalFee			 = 5000 * 1000000; //tx.rentalFee;
			var namespaceParent		 = tx.namespaceParent ? tx.namespaceParent.fqn : null;
			var namespaceName		 = tx.namespaceName.toString();
			var due					 = tx.due;
			var entity				 = obj.constructNamespace(actualSender, rentalFeeSink, rentalFee, namespaceParent, namespaceName, due);
			if (tx.isMultisig)
			{
				entity				 = this.multisigWrapper(kp.publicKey.toString(), entity, due);
			}
			return entity;
		};


		/**
		 * TransactionsFactory.prepareMosaicDefinition
		 * ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @return
		 */
		obj.prepareMosaicDefinition = function(privatekey, tx)
		{
			var keyPair				 = new KeyPair();
			var kp					 = keyPair.create(privatekey);
			var actualSender		 = tx.isMultisig ? tx.multisigAccount.publicKey : kp.publicKey.toString();
			var rentalFeeSink		 = tx.mosaicFeeSink.toString();
			var rentalFee			 = tx.mosaicFee;
			var namespaceParent		 = tx.namespaceParent.fqn;
			var mosaicName			 = tx.mosaicName.toString();
			var mosaicDescription	 = tx.mosaicDescription.toString();
			var mosaicProperties	 = tx.properties;
			var levy				 = tx.levy.mosaic ? tx.levy : null;
			var due					 = tx.due;
			var entity				 = this.constructMosaicDefinition(actualSender, rentalFeeSink, rentalFee, namespaceParent, mosaicName, mosaicDescription, mosaicProperties, levy, due);
			if (tx.isMultisig)
			{
				entity				 = this.multisigWrapper(kp.publicKey.toString(), entity, due);
			}
			return entity;
		};


		/**
		 * TransactionsFactory.prepareMosaicSupply
		 * ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @return
		 */
		obj.prepareMosaicSupply = function(privatekey, tx)
		{
			var keyPair				 = new KeyPair();
			var kp					 = keyPair.create(privatekey);
			var actualSender		 = tx.isMultisig ? tx.multisigAccount.publicKey : kp.publicKey.toString();
			var due					 = tx.due;
			var entity				 = this.constructMosaicSupply(actualSender, tx.mosaic, tx.supplyType, tx.delta, due);
			if (tx.isMultisig)
			{
				entity				 = this.multisigWrapper(kp.publicKey.toString(), entity, due);
			}
			return entity;
		};


		/**
		 * TransactionsFactory.prepareSignature
		 * ...
		 * @param privateke			 ...
		 * @param tx				 ...
		 * @param nisPort			 ...
		 * @param cb				 ...
		 * @param failedCb			 ...
		 * @return
		 */
		obj.prepareSignature = function(privatekey, tx, nisPort, cb, failedCb)
		{
			var convert				 = new Convert();
			var keyPair				 = new KeyPair();
			var kp					 = keyPair.create(fixPrivateKey(privatekey));
			var actualSender		 = kp.publicKey.toString();
			var otherAccount		 = tx.multisigAccountAddress.toString();
			var otherHash			 = tx.hash.toString();
			var due					 = tx.due;
			var entity				 = this.constructSignature(actualSender, otherAccount, otherHash, due);
			var result				 = this.serializeTransaction(entity);
			var signature			 = kp.sign(result);
			var obj = {
				'data':convert.ua2hex(result),
				'signature':signature.toString()
			};
			$http.post('http://'+CURRENT_HOSTNAME+':'+nisPort+'/transaction/announce', obj).then(function (data)
			{
				cb(data);
			}, function(data) {
				failedCb('announce', data);
			});
		};


		/**
		 * TransactionsFactory.prepareSignature
		 * ...
		 * @param privateke			 ...
		 * @param tx				 ...
		 * @param nisPort			 ...
		 * @param cb				 ...
		 * @param failedCb			 ...
		 * @return
		 */
		obj.prepareSignatureMultisigTransaction = function(privatekey, tx)
		{
			var convert				 = new Convert();
			var keyPair				 = new KeyPair();
			var kp					 = keyPair.create(fixPrivateKey(privatekey));
			var actualSender		 = kp.publicKey.toString();
			var otherAccount		 = tx.multisigAccountAddress.toString();
			var otherHash			 = tx.hash.toString();
			var due					 = tx.due;
			var entity				 = this.constructSignature(actualSender, otherAccount, otherHash, due);
			return entity;
		};


		/**
		 * TransactionsFactory.prepareConvertMultisigAccount
		 * ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @return
		 */
		obj.prepareMultisigAggregateModification = function(privatekey, tx)
		{
			var keyPair				 = new KeyPair();
			var kp					 = keyPair.create(privatekey);
			var actualSender		 = tx.isMultisig ? tx.multisigPubKey : kp.publicKey.toString();
			var multisigPubKey		 = tx.multisigPubKey;
			var due					 = tx.due;
			var modifications		 = tx.cosignatoryAccount;
			var minCosignatories	 = tx.minCosignatories;
			var entity				 = obj.constructMultisigAggregateModification(multisigPubKey, modifications, minCosignatories, due);
			if (tx.isMultisig)
			{
				entity				 = this.multisigWrapper(kp.publicKey.toString(), entity, due);
			}
			return entity;
		};


		/**
		 * TransactionsFactory.serializeAndAnnounceTransaction
		 * ...
		 * @param entity			 ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @param nisPort			 ...
		 * @param cb				 ...
		 * @param failedCb			 ...
		 * @return
		 */
		obj.serializeAndAnnounceTransaction = function(entity, privatekey, tx, nisPort, cb, failedCb)
		{
			var convert		 = new Convert();
			var keyPair		 = new KeyPair();
			var kp			 = keyPair.create(fixPrivateKey(privatekey));
			var result		 = this.serializeTransaction(entity);
			var signature	 = kp.sign(result);
			var obj			 = {
				'data':convert.ua2hex(result),
				'signature':signature.toString()
			};
			$http.post('http://'+CURRENT_HOSTNAME+':'+nisPort+'/transaction/announce', obj).then(function (data)
			{
				cb(data);
			}, function(data) {
				failedCb('announce', data);
			});
		};


		/**
		 * TransactionsFactory.serializeAnnounceTransaction
		 * ... 
		 * @param entity			 ...
		 * @param privatekey		 ...
		 * @param tx				 ...
		 * @return
		 */
		obj.serializeAnnounceTransaction = function(entity, privatekey, tx)
		{
			var convert		 = new Convert();
			var keyPair		 = new KeyPair();
			var kp			 = keyPair.create(fixPrivateKey(privatekey));
			var result		 = this.serializeTransaction(entity);
			var signature	 = kp.sign(result);
			var obj			 = {
				'data':convert.ua2hex(result),
				'signature':signature.toString()
			};
			return obj;
		};

		return obj;
	}


	/**
	 * KeyPair
	 * ... utils/KeyPair.js から作成
	 * @return
	 */
	var KeyPair = function()
	{
		var obj = {};

		/**
		 * KeyPair.create
		 * ...
		 * @param hexdata			 ...
		 * @return
		 */
		obj.create = function(hexdata)
		{
			var r		 = new KEYPAIR(hexdata);
			return r;
		}

		return obj;
	};


	/**
	 * CryptoHelpers
	 * ... utils/CryptoHelpers.js から作成
	 * @return
	 */
	var CryptoHelpers = function()
	{
		var obj = {};

		/**
		 * CryptoHelpers.generateKey
		 * ...
		 * @param salt					 ...
		 * @param password				 ...
		 * @param numberOfIterations	 ...
		 * @return
		 */
		obj.generateKey = function(salt, password, numberOfIterations)
		{
			console.time('pbkdf2 generation time');
			var key256Bits = CryptoJS.PBKDF2(password, salt, {
					keySize:		 256/32,
					iterations:		 numberOfIterations,
					hasher:			 CryptoJS.algo.SHA256
			});
			console.timeEnd('pbkdf2 generation time');
			return {
				'salt':CryptoJS.enc.Hex.stringify(salt),
				'priv':CryptoJS.enc.Hex.stringify(key256Bits)
			};
		};


		/**
		 * CryptoHelpers.generateSaltedKey
		 * ...
		 * @param salt					 ...
		 * @param password				 ...
		 * @param numberOfIterations	 ...
		 * @return
		 */
		obj.generateSaltedKey = function(password)
		{
			var salt		 = CryptoJS.lib.WordArray.random(128/8);
			return this.generateKey(salt, password, 1000);
		};


		/**
		 * CryptoHelpers.derivePassSha
		 * ...
		 * @param password				 ...
		 * @param password				 ...
		 * @return
		 */
		obj.derivePassSha = function(password, count)
		{
			var data		 = password;
			console.time('sha3^n generation time');
			for (var i = 0; i < count; ++i)
			{
				data	 = CryptoJS.SHA3(data, {
												outputLength: 256
											});
			}
			console.timeEnd('sha3^n generation time');
			var r			 = {
									'priv':		CryptoJS.enc.Hex.stringify(data)
								};
			return r;
		};


		/**
		 * CryptoHelpers.passwordToPrivatekeyClear
		 * ...
		 * @param commonData			 ...
		 * @param walletAccount			 ...
		 * @param doClear				 ...
		 * @return
		 */
		obj.passwordToPrivatekeyClear = function(commonData, walletAccount, doClear)
		{
			if (commonData.password)
			{
				var r			 = undefined;
				if (walletAccount.algo === "pass:6k")
				{
					r			 = this.derivePassSha(commonData.password, 6000);
				} else if (walletAccount.algo === "pbkf2:1k") {
					r			 = this.generateKey(CryptoJS.enc.Hex.parse(walletAccount.salt), commonData.password);
				} else if (walletAccount.algo === "pass:enc") {
					var pass	 = this.derivePassSha(commonData.password, 20);
					var obj		 = {
						ciphertext:			CryptoJS.enc.Hex.parse(walletAccount.encrypted),
						iv:					convert.hex2ua(walletAccount.iv),
						key:				convert.hex2ua(pass.priv)
					}
					var d		 = this.decrypt(obj);
					r			 = {
						'priv':d
					};
				} else {
					alert("unknown wallet encryption method");
				}
				if (doClear) {
					delete commonData.password;
				}
				return r.priv;
			} else {
				return commonData.privatekey;
			}
		}


		/**
		 * CryptoHelpers.generateSaltedKey
		 * ...
		 * @param priv					 ...
		 * @param network				 ...
		 * @param _expectedAddress		 ...
		 * @return
		 */
		obj.checkAddress = function(priv, network, _expectedAddress)
		{
			var keyPair				 = new KeyPair();
			var Address				 = new Address();
			var expectedAddress		 = _expectedAddress.toUpperCase().replace(/-/g, '');
			var kp					 = keyPair.create(priv);
			var address				 = Address.toAddress(kp.publicKey.toString(), network);
			return address === expectedAddress;
		};


		/**
		 * CryptoHelpers.passwordToPrivatekey
		 * ...
		 * @param commonData			 ...
		 * @param networkId				 ...
		 * @param walletAccount			 ...
		 * @return
		 */
		obj.passwordToPrivatekey = function(commonData, networkId, walletAccount)
		{
			var priv = this.passwordToPrivatekeyClear(commonData, walletAccount, false);
			if (!this.checkAddress(priv, networkId, walletAccount.address))
			{
				return false;
			}
			commonData.privatekey = priv;
			return true;
		};


		/**
		 * CryptoHelpers.randomKey
		 * ...
		 * @return
		 */
		obj.randomKey = function randomKey()
		{
			var rkey			 = new Uint8Array(32);
			window.crypto.getRandomValues(rkey);
			return rkey;
		};


		/**
		 * CryptoHelpers.encrypt
		 * ...
		 * @param data				 ...
		 * @param key				 ...
		 * @return
		 */
		obj.encrypt = function encrypt(data, key)
		{
			var iv			 = new Uint8Array(16);
			window.crypto.getRandomValues(iv);
			var encKey		 = ua2words(key, 32);
			var encIv		 = {
									iv: ua2words(iv, 16)
								};
			var encrypted	 = CryptoJS.AES.encrypt(CryptoJS.enc.Hex.parse(data), encKey, encIv);
			return {
				ciphertext: encrypted.ciphertext,
				iv:iv,
				key:key
			};
		};


		/**
		 * CryptoHelpers.decrypt
		 * ...
		 * @param data					 ...
		 * @return hex string
		 */
		obj.decrypt = function decrypt(data)
		{
			var encKey			 = ua2words(data.key, 32);
			var encIv			 = {
										iv: ua2words(data.iv, 16)
									};
			return CryptoJS.enc.Hex.stringify(CryptoJS.AES.decrypt(data, encKey, encIv));
		};


		/**
		 * CryptoHelpers.encodePrivKey
		 * ...
		 * @param privatekey			 ...
		 * @param password				 ...
		 * @return
		 */
		obj.encodePrivKey = function encodePrivKey(privatekey, password)
		{
			var convert			 = new Convert();
			var pass			 = this.derivePassSha(password, 20);
			var r				 = this.encrypt(privatekey, convert.hex2ua(pass.priv));
			var ret				 = {
										ciphertext:CryptoJS.enc.Hex.stringify(r.ciphertext), iv:convert.ua2hex(r.iv)
									};
			return ret;
		};


		/**
		 * CryptoHelpers.encode
		 * ...
		 * @param senderPriv			 ...
		 * @param recipientPub			 ...
		 * @param msg					 ...
		 * @return
		 */
		obj.encode = function(senderPriv, recipientPub, msg)
		{
			var convert				 = new Convert();
			var iv					 = new Uint8Array(16);
			window.crypto.getRandomValues(iv);
			var sk					 = convert.hex2ua_reversed(senderPriv);
			var pk					 = convert.hex2ua(recipientPub);
			var salt				 = new Uint8Array(32);
			window.crypto.getRandomValues(salt);
			var shared				 = new Uint8Array(32);
			var r					 = key_derive(shared, salt, sk, pk);
			var encKey				 = r;
			var encIv				 = {
											iv: ua2words(iv, 16)
										};
			var encrypted			 = CryptoJS.AES.encrypt(CryptoJS.enc.Hex.parse(convert.utf8ToHex(msg)), encKey, encIv);
			var result				 = convert.ua2hex(salt) + convert.ua2hex(iv) + CryptoJS.enc.Hex.stringify(encrypted.ciphertext);
			return result;
		};


		/**
		 * CryptoHelpers.decode
		 * ...
		 * @param recipientPrivate			 ...
		 * @param senderPublic				 ...
		 * @param payload					 ...
		 * @return
		 */
		obj.decode = function(recipientPrivate, senderPublic, payload)
		{
			var convert				 = new Convert();
			var binPayload			 = convert.hex2ua(payload);
			var salt				 = new Uint8Array(binPayload.buffer, 0, 32);
			var iv					 = new Uint8Array(binPayload.buffer, 32, 16);
			var payload				 = new Uint8Array(binPayload.buffer, 48);
			var sk					 = convert.hex2ua_reversed(recipientPrivate);
			var pk					 = convert.hex2ua(senderPublic);
			var shared				 = new Uint8Array(32);
			var r					 = key_derive(shared, salt, sk, pk);
			var encKey				 = r;
			var encIv				 = {
											iv: ua2words(iv, 16)
										};
			var encrypted			 = {
											'ciphertext':ua2words(payload,payload.length)
										};
			var plain				 = CryptoJS.AES.decrypt(encrypted, encKey, encIv);
			var hexplain			 = CryptoJS.enc.Hex.stringify(plain);
			return hexplain;
		};

		return obj;
	}


	/**
	 * Address
	 * ... utils/Address.js から作成
	 * @return
	 */
	var Address = function()
	{
		var obj			 = {};

		/**
		 * Address.toAddress
		 * ...
		 * @param publicKey				 ...
		 * @param networkId				 ...
		 * @return
		 */
		obj.toAddress = function toAddress(publicKey, networkId)
		{
			var convert				 = new Convert();
			var binPubKey			 = CryptoJS.enc.Hex.parse(publicKey);
			var hash				 = CryptoJS.SHA3(binPubKey, {
										outputLength: 256
									});
			var hash2				 = CryptoJS.RIPEMD160(hash);
			// 98 is for testnet
			var networkPrefix		 = (networkId === -104) ? '98' : (networkId === 104 ? '68' : '60');
			var versionPrefixedRipemd160Hash	 = networkPrefix + CryptoJS.enc.Hex.stringify(hash2);
			var tempHash						 = CryptoJS.SHA3(CryptoJS.enc.Hex.parse(versionPrefixedRipemd160Hash), {
					outputLength: 256
				});
			var stepThreeChecksum				 = CryptoJS.enc.Hex.stringify(tempHash).substr(0, 8);
			var concatStepThreeAndStepSix		 = convert.hex2a(versionPrefixedRipemd160Hash + stepThreeChecksum);
			var ret								 = baseenc.b32encode(concatStepThreeAndStepSix);
			return ret;
		};


		/**
		 * Address.isFromNetwork
		 * ...
		 * @param _address				 ...
		 * @param networkId				 ...
		 * @return
		 */
		obj.isFromNetwork = function isFromNetwork(_address, networkId)
		{
			var address			 = _address.toString().toUpperCase().replace(/-/g, '');
			var a				 = address[0];
			return (networkId === -104 && a === 'T') || (networkId === 104 && a === 'N') || (networkId === 96 && a === 'M');
		};


		/**
		 * Address.isValid
		 * ...
		 * @param recipientPrivate			 ...
		 * @param senderPublic				 ...
		 * @param payload					 ...
		 * @return
		 */
		obj.isValid = function isValid(_address)
		{
			var convert				 = new Convert();
			var address			 = _address.toString().toUpperCase().replace(/-/g, '');
			if (!address || address.length !== 40)
			{
				return false;
			}
			var decoded			 = convert.ua2hex(baseenc.b32decode(address));
			var versionPrefixedRipemd160Hash	 = CryptoJS.enc.Hex.parse(decoded.slice(0, 42));
			var tempHash						 = CryptoJS.SHA3(versionPrefixedRipemd160Hash, {
				outputLength: 256
			});
			var stepThreeChecksum				 = CryptoJS.enc.Hex.stringify(tempHash).substr(0, 8);
			return stepThreeChecksum === decoded.slice(42);
		};

		return obj;
	}


	/**
	 * SessionData
	 * ... services/SessionData.js から作成
	 *
	 * @access
	 * @return
	 */
	var SessionData = function()
	{
		var networkId		 = undefined;
		var nisPort			 = 0;
		var rememberedKey	 = undefined;
		var nisNode			 = undefined;

		return {
			setNetworkId: function setNetworkId(id)
			{
				networkId	 = id;
			},
			getNetworkId: function getNetworkId()
			{
				return networkId;
			},
			setNode: function setNode(node)
			{
				nisNode		 = node;
			},
			getNode: function getNode()
			{
				return nisNode;
			},
			setNisPort: function setNisPort(port)
			{
				nisPort		 = port;
			},
			getNisPort: function getNisPort()
			{
				return nisPort;
			},
			setRememberedKey: function setRememberedKey(data)
			{
				rememberedKey	 = data
			},
			getRememberedKey: function getRememberedKey()
			{
				return rememberedKey;
			}
		};
	}


	//
	// WalletApp return
	//
	return {
		createAnnounceJSON: function(publickey,privatekey,data)
		{
			var transactionsFactory		 = new TransactionsFactory();
			var entity					 = '';
			switch (data.prepare)
			{
				case 'prepareTransferV1':
					entity				 = transactionsFactory.prepareTransfer(publickey,privatekey,data);
					break;
				case 'prepareTransferV2':
					entity				 = transactionsFactory.prepareTransferV2(privatekey,data);
					break;
				case 'prepareMultisigAggregateModification':
					entity				 = transactionsFactory.prepareMultisigAggregateModification(privatekey, data);
					break;
				case 'prepareSignatureMultisigTransaction':
					entity				 = transactionsFactory.prepareSignatureMultisigTransaction(privatekey, data);
					break;
				case 'prepareNamespace':
					entity				 = transactionsFactory.prepareNamespace(privatekey,data);
					break;
				case 'prepareMosaicDefinition':
					entity				 = transactionsFactory.prepareMosaicDefinition(privatekey,data);
					break;
				default:
					break;
			}
			var convertData				 = transactionsFactory.serializeAnnounceTransaction(entity,privatekey,data);
			return JSON.stringify(convertData);
		}
	};

}();


